;Title:				Four Channel PWM Controller
;					AVR Design Contest 2004

;Entrant:			A3773

;Date:				040526
;Submitted:			040628
;Last Modified:		040701

;Target Device(s):	ATMega16,ATMega16L,ATMega32,ATMega32L

;Description:		A four channel PWM controller. Target application is flexible.
;					PWM control is selectable via analog input, keypad, or RS232
;					interface. Three general purpose outputs are available. Four
;					outputs control indicators of manual PWM control, or optionally
;					are user controlled. An LCD interface is provided.

;Notes:				Set the target device before compile (this file).
;					Set the xtal frequency before compile (Labels.asm).
;					Set the baud rate before compile (Labels.asm)
;					Set LCD startup message before compile (EEPROM Values.asm).
;					Set PWM channel names before compile (EEPROM Values.asm).
;					Set ADC channel names before compile (EEPROM Values.asm).

;Fuse Bits Programmed:
;					OCD disabled
;					SPIEN
;					EESAVE (to retain EEPROM values during prototyping)
;					CKOPT,CKSEL3:1=1,111
;					CKSEL0,SUT1:0=1,11


;Target Device Include
.include "m32def.inc"
.include "m32def_missings.asm"

;Define Labels
.include "Labels.asm"

;Define RAM Variables
.include "RAM Variables.asm"

;Define EEPROM values
.include "EEPROM Values.asm"

.listmac

;Program Start
.cseg
.org 0x0000

;Reset and interrupt vectors

jmp RESET

.org INT0addr
jmp EXT_INT

.org INT1addr
jmp EXT_INT

.org INT2addr
jmp EXT_INT

.org OVF0addr
jmp TMR0_OVF

.org URXCaddr
jmp USART_RXC

.org UDREaddr
jmp USART_DRE

.org ADCCaddr
jmp ADC_INT

.org ERDYaddr
jmp EE_RDY

.org 0x002A

;Text Messages
.include "Text Messages.asm"


;FIFO Routines
.include "FIFO Routines.asm"

;LCD Routines
.include "LCD Routines.asm"

;Keypad Routines
.include "Keypad Routines.asm"

;RS232 Control Byte Routines
.include "RS232 Control Byte Routines.asm"

;BCD Math
.include "BCD Math.asm"

;Interrupt Service Routines
.include "Interrupt Service Routines.asm"

;Device Initialisation
RESET:	ldi r16,high(RAMEND)	;Set stack pointer
		out SPH,r16
		ldi r16,low(RAMEND)
		out SPL,r16

		ldi r16,(1<<ACD)		;disable analog comparator
		out ACSR,r16

		;Read Control bytes from EEPROM
		ldi ZL,low(EE_DC1)
		ldi ZH,high(EE_DC1)		;set EE address for DC1

		out EEARL,ZL
		out EEARH,ZH

		sbi EECR,EERE

		in DC1_B0,EEDR			;load DC1_B0

		adiw ZH:ZL,1			;set EE address for DC2

		out EEARL,ZL
		out EEARH,ZH

		sbi EECR,EERE

		in DC2_B0,EEDR			;load DC2_B0

		adiw ZH:ZL,1			;set EE address for DC3

		out EEARL,ZL
		out EEARH,ZH

		sbi EECR,EERE

		in DC3_B0,EEDR			;load DC3_B0

		adiw ZH:ZL,1			;set EE address for DC4

		out EEARL,ZL
		out EEARH,ZH

		sbi EECR,EERE

		in DC4_B0,EEDR			;load DC4_B0

		;Read PWM and ADC Names from EEPROM
		ldi r17,88				;set counter

		ldi ZL,low(EE_PWM0_Name)
		ldi ZH,high(EE_PWM0_Name)	;Set initial address

		ldi YL,low(PWM0_Name)
		ldi YH,high(PWM0_Name)		;Set SRAM address

EE_Name_Loop:
		out EEARL,ZL
		out EEARH,ZH

		sbi EECR,EERE

		in r16,EEDR				;load data into r16

		st Y+,r16				;Save data and post inc sram address

		adiw ZH:ZL,1			;inc address

		dec r17					;dec counter

		brne EE_Name_Loop


		;Timer Initial PWM Load
		ldi ZL,low(EE_PWM0)
		ldi ZH,high(EE_PWM0)	;set EE address for PWM0

		out EEARL,ZL
		out EEARH,ZH

		sbi EECR,EERE

		in r12,EEDR				;load PWM0

		adiw ZH:ZL,1			;set EE address for PWM1

		out EEARL,ZL
		out EEARH,ZH

		sbi EECR,EERE

		in r13,EEDR				;load PWM1

		adiw ZH:ZL,1			;set EE address for PWM2

		out EEARL,ZL
		out EEARH,ZH

		sbi EECR,EERE

		in r14,EEDR				;load PWM2

		adiw ZH:ZL,1			;set EE address for PWM3

		out EEARL,ZL
		out EEARH,ZH

		sbi EECR,EERE

		in r15,EEDR				;load PWM3

		;Timer0 Setup
		out OCR0,r12			;Compare 1 value = PWM0

		ldi r16,(1<<WGM00 | 1<<WGM01 | 1<<COM01 | 1<<CS00)
		out TCCR0,r16			;Waveform:		Fast PWM
								;Compare:		Clear OC0 on match
								;Clock:			FOsc, 1:1

		in r16,TIMSK
		sbr r16,(1<<TOIE0)
		out TIMSK,r16			;TMR0 Overflow:	Interrupt enabled
		
		;Timer1 Setup
		clr r16
		out OCR1AH,r16
		out OCR1AL,r13			;Compare A value = PWM1
		out OCR1BH,r16
		out OCR1BL,r14			;Compare B value = PWM2

		out ICR1H,r16
		ldi r16,0xFF
		out ICR1L,r16			;ICR1 = 0x00FF

		ldi r16,(1<<COM1A1 | 1<<COM1B1 | 1<<WGM11)
		out TCCR1A,r16			;Waveform:		Fast PWM,ICR1 as TOP (WGM=14)
												;(WGM=5 gives improper simulation)
								;Compare A:		Clear OC1A on match
								;Compare B:		Clear OC1B on match
		
		ldi r16,(1<<WGM13 | 1<<WGM12 | 1<<CS10)
		out TCCR1B,r16			;Clock:			FOsc, 1:1


		;Timer2 Setup
		out OCR2,r15			;Comapre 2 Value = PWM3

		ldi r16,(1<<WGM20 | 1<<WGM21 | 1<<COM21 | 1<<CS20)
		out TCCR2,r16			;Waveform:		Fast PWM
								;Compare:		Clear OC0 on match
								;Clock:			FOsc, 1:1

		;Timer Synchronisation
		clr r16
		out TCNT0,r16			;Timer0 = 0

		ldi r16,0x02
		out TCNT2,r16			;Timer2 = 2

		clr r17
		ldi r16,0x06
		out TCNT1H,r17
		out TCNT1L,r16			;Timer1 = 6


		;Port Setup
		ldi r16,0b11110000		;PORTA<7:4> = output
								;PORTA<3:0> = input
		out DDRA,r16

		ldi r16,0b11111011		;PORTB<7,6,5,4,3,1,0> = output 
								;PORTB<2> = input
		out DDRB,r16

		ldi r16,0b11111111		;PORTC<7:0> = output (changes for LCD read)
		out DDRC,r16

		ldi r16,0b11110010		;PORTD<7,6,5,4,1> = output
								;PORTD<3,2,0> = input
		out DDRD,r16

		clr r16					;PORTA,C all pullups off, outputs low
		out PORTA,r16
		out PORTC,r16
		
		ldi r16,0b00000100		;PORTB<3> pullup on, outputs all low
		out PORTB,r16

		ldi r16,0b00001100		;PORTD<3,2> pullups on, outputs all low
		out PORTD,r16

		call Set_BKL			;Set BKL and GPO pins. IND pins set during ADC ISR
		call Set_GPO


		;USART Setup
		ldi r16,low(UBRR_Val)
		ldi r17,high(UBRR_Val)

		out UBRRH,r17
		out UBRRL,r16			;Baud Rate:		UBRR_Val

		ldi r16,(1<<UDRE)
		out UCSRA,r16			;Double Speed:	Off
								;MPCM:			Off

		ldi r16,(1<<RXCIE | 1<<RXEN | 1<<TXEN)
		out UCSRB,r16			;UX Int:		On
								;TC Int:		Off
								;DR Int:		Off
								;Receive:		Enable
								;Transmit:		Enable
		
		ldi r16,(1<<URSEL | 1<<UCSZ1 | 1<<UCSZ0)
		out UCSRC,r16			;Mode:			Asynchronous
								;Parity:		None
								;Stop:			1 bit
								;Data:			8 bit

		;ADC Setup
		ldi r16,(1<<ADLAR)
		out ADMUX,r16			;Reference:		AREF
								;Output:		Left Adjust
								;Channel:		ADC0

		ldi r16,(1<<ADEN | 1<<ADIE | ADC_Prescale_Val)
		out ADCSR,r16			;ADC:			Enabled
								;Auto Trigger:	Off
								;ADC Int.:		Enabled
								;Prescale:		ADC_Prescale_Val


		;External Interrupt Setup

		in r16,GICR
		cbr r16,(1<<INT0 | 1<<INT1 | 1<<INT2)
		out GICR,r16			;disable INT0,1,2 interrupts

		in r16,MCUCR
		sbr r16,(1<<ISC11 | 1<<ISC01)
		out MCUCR,r16			;INT0 trigger:	falling edge
								;INT1 trigger:	falling edge

		in r16,MCUCSR
		cbr r16,(1<<ISC2)
		out MCUCSR,r16			;INT2 trigger:	falling edge

		ldi r16,(1<<INTF1 | 1<<INTF0 | 1<<INTF2)
		out GIFR,r16			;clear INT0,1,2 flags

		in r16,GICR
		sbr r16,(1<<INT0 | 1<<INT1 | 1<<INT2)
		out GICR,r16			;Enable INT0,1,2 interrupts

	
		;Variable Initialisation

		Set_FIFO_Address LCD_FIFO
		ldi r16,LCD_FIFO_Size
		call FIFO_Init

		Set_FIFO_Address Keypad_FIFO
		ldi r16,Keypad_FIFO_Size
		call FIFO_Init

		Set_FIFO_Address EEPROM_FIFO
		ldi r16,EEPROM_FIFO_Size
		call FIFO_Init

		Set_FIFO_Address USART_RX_FIFO
		ldi r16,USART_RX_FIFO_Size
		call FIFO_Init

		Set_FIFO_Address USART_TX_FIFO
		ldi r16,USART_TX_FIFO_Size
		call FIFO_Init

		ldi r16,low(Read_USART_RX_FIFO_Default)
		ldi r17,high(Read_USART_RX_FIFO_Default)
		sts Read_USART_Jump_Adr,r16
		sts (Read_USART_Jump_Adr+1),r17

		ldi r16,(1<<Update_LCD0 | 1<<Test_LCD0 | 1<<Update_LCD1 | 1<<Test_LCD1)
		mov Ticker_Flags,r16	;set initial updating and testing

		clr Keypad_C0L
		clr Keypad_C0H			;Keypad counter = 0
		clr LCD_C0				;LCD0 counter = 0

		ldi r16,low(1);31250)
		ldi r17,high(1);31250)

		mov LCD_C1L,r16
		mov LCD_C1H,r17			;LCD1 counter = 1s

		ldi r16,(1<<Test_RX_FIFO | 1<<Test_TX_FIFO | 1<<Test_EEPROM_FIFO)
		mov Immediate_Flags,r16	;Enable all tests

		ldi r16,low(LCD0_Routine_Default)
		ldi r17,high(LCD0_Routine_Default)
		sts LCD0_Routine_Jump_Adr,r16
		sts LCD0_Routine_Jump_Adr+1,r17		;Default LCD0 Jump Address

		ldi r16,low(Keypad_Scan_Input_Wait)
		ldi r17,high(Keypad_Scan_Input_Wait)
		
		sts Keypad_Scan_Jump_Adr,r16
		sts Keypad_Scan_Jump_Adr+1,r17		;Initial Seypad_Scan jump address

		ldi r16,low(LCD1_Routine_Init_Delay)
		ldi r17,high(LCD1_Routine_Init_Delay)
		sts LCD1_Routine_Jump_Adr,r16
		sts LCD1_Routine_Jump_Adr+1,r17		;Initial LCD1 Jump Address

		ldi r16,4
		sts LCD1_Routine_Counter,r16		;Initial LCD1 counter value

		ldi r16,(1<<Disp_ADC)
		sts LCD1_Routine_Flags,r16				;Initialise LCD1 flags



		;Clear Interrupt Flags
		in r16,TIFR
		sbr r16,(1<<TOV0)
		out TIFR,r16			;TMR0 Overflow:	Cleared

		;Enable Interrupts
		sei

		;Start ADC conversions
		in r16,ADCSR
		sbr r16,(1<<ADSC)
		out ADCSR,r16

		;Setup LCD
		call LCD_Init

		call LCD_SMsg

		;Clear Ticker_Val before running Main Loop
		clr Ticker_Val

MAIN:	sbrs Immediate_Flags,Test_RX_FIFO
		rjmp MAIN_A

		Set_FIFO_Address USART_RX_FIFO
		ld r16,Z				;Get RX Usart counter value
		cpi r16,0x00
		breq MAIN_A
		call Read_USART_RX_FIFO	;Read USART RX FIFO if there are waiting bytes

MAIN_A:	sbrs Immediate_Flags,Test_TX_FIFO
		rjmp MAIN_B

		Set_FIFO_Address USART_TX_FIFO
		ld r16,Z				;Get TX Usart counter value
		cpi r16,0x00
		breq MAIN_B 			;Start transmission if counter>0

		mov r16,Immediate_Flags
		cbr r16,(1<<Test_TX_FIFO)
		mov Immediate_Flags,r16	;Stop testing TX FIFO until cleared

		in r16,UCSRB
		sbr r16,(1<<UDRIE)
		out UCSRB,r16			;Enable Data Empty Interrupt

MAIN_B: sbrs Immediate_Flags,Test_EEPROM_FIFO
		rjmp MAIN_C

		Set_FIFO_Address EEPROM_FIFO
		ld r16,Z				;Get EEPROM FIFO counter value
		cpi r16,0x00
		breq MAIN_C				;Start writing if counter>0
		
		;Begin EEPROM writing
		mov r16,Immediate_Flags
		cbr r16,(1<<Test_EEPROM_FIFO)
		mov Immediate_Flags,r16	;Stop testing EEPROM FIFO until cleared

		in r16,EECR
		sbr r16,(1<<EERIE)
		out EECR,r16			;Enable EE Ready Interrupt

MAIN_C:	mov r16,Ticker_Val		;Get Ticker_Val

		clr Ticker_Val			;reset Ticker_Val

		clr r17					;r17:r16 = Ticker_Val

		sbrs Ticker_Flags,Update_Keypad
		rjmp MAIN_C2

		sub Keypad_C0L,r16
		sbc Keypad_C0H,r17		;lower counter

MAIN_C2:
		sbrs Ticker_Flags,Test_Keypad
		rjmp MAIN_C3

		brsh MAIN_C3			;Do Keypad routine if counter<0

		push r16
		push r17				;save Ticker_Val

		call Keypad_Scan

		pop r17
		pop r16					;restore Ticker_Val

MAIN_C3:
		sbrs Ticker_Flags,Update_LCD0
		rjmp MAIN_C4

		sub LCD_C0,r16		;lower counter

MAIN_C4:
		sbrs Ticker_Flags,Test_LCD0
		rjmp MAIN_C5

		brsh MAIN_C5			;Do LCD0 routine if counter<0

		push r16
		push r17				;save Ticker_Val

		call LCD0_Routine

		pop r17
		pop r16					;restore Ticker_Val

MAIN_C5:
		sbrs Ticker_Flags,Update_LCD1
		rjmp MAIN_C6

		sub LCD_C1L,r16
		sbc LCD_C1H,r17		;lower counter

MAIN_C6:
		sbrs Ticker_Flags,Test_LCD1
		rjmp MAIN_D

		brsh MAIN_D			;Do LCD1 routine if counter<0

		call LCD1_Routine
		

MAIN_D:

		rjmp MAIN





